//
//  KeyCommands.swift
//  Do It
//
//  Created by Jim Dovey on 4/1/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import UIKit
import SwiftUI
import Combine

struct KeyCommand: Hashable {
    private let _command: UIKeyCommand
    
    fileprivate init?(_ uikit: UIKeyCommand?) {
        guard let uikit = uikit else { return nil }
        self._command = uikit
    }
    
    init(title: String, input: String, modifierFlags: ModifierFlags = [],
         attributes: Attributes = [], discoverabilityTitle: String? = nil) {
        self._command = UIKeyCommand(
            title: title,
            action: #selector(UIResponder.swiftUIKeyCommand(_:)),
            input: input,
            modifierFlags: UIKeyModifierFlags(rawValue: modifierFlags.rawValue),
            discoverabilityTitle: discoverabilityTitle,
            attributes: UIMenuElement.Attributes(rawValue: attributes.rawValue)
        )
    }
    
    fileprivate var uikit: UIKeyCommand { _command }
    
    var input: String? { _command.input }
    var modifierFlags: UIKeyModifierFlags { _command.modifierFlags }
    var title: String { _command.title }
    var discoverabilityTitle: String? { _command.discoverabilityTitle }
    
    struct ModifierFlags: OptionSet {
        var rawValue: Int
        
        init(rawValue: Int) {
            self.rawValue = rawValue
        }
        private init(_ uikit: UIKeyModifierFlags) {
            self.rawValue = uikit.rawValue
        }
        
        static var alphaShift = ModifierFlags(.alphaShift)
        static var shift = ModifierFlags(.shift)
        static var control = ModifierFlags(.control)
        static var alternate = ModifierFlags(.alternate)
        static var command = ModifierFlags(.command)
        static var numericPad = ModifierFlags(.numericPad)
    }
    
    struct Attributes: OptionSet {
        var rawValue: UInt
        
        init(rawValue: UInt) {
            self.rawValue = rawValue
        }
        private init(_ uikit: UIMenuElement.Attributes) {
            self.rawValue = uikit.rawValue
        }
        
        static var disabled = Attributes(.disabled)
        static var destructive = Attributes(.destructive)
        static var hidden = Attributes(.hidden)
    }
    
    static func == (lhs: KeyCommand, rhs: KeyCommand) -> Bool {
        return lhs._command == rhs._command
    }
    
    func hash(into hasher: inout Hasher) {
        hasher.combine(_command)
    }
}


// Defines the action targeted by SwiftUI key commands, so that
// #selector() works.
fileprivate extension UIResponder {
    @objc func swiftUIKeyCommand(_ sender: UIKeyCommand?) {}
}

// MARK: -

// START:CommandRegistrar
fileprivate final class CommandRegistrar {
    typealias CommandPublisher = PassthroughSubject<KeyCommand,Never>
    var publisher = PassthroughSubject<KeyCommand,Never>()
    
    var commands: Set<KeyCommand> = []
    
    // START:InstallCommand
    func install(command: KeyCommand) -> AnyPublisher<KeyCommand, Never> {
        // END:InstallCommand
        // <literal:elide> ... </literal:elide>
        // START:InstallCommand
        // END:CommandRegistrar
        commands.insert(command)
        return publisher
            .filter { $0 == command }
            .eraseToAnyPublisher()
        // START:CommandRegistrar
    }
    // END:InstallCommand
    
    func remove(command: KeyCommand) {
        commands.remove(command)
    }
}
// END:CommandRegistrar

// START:Controller
fileprivate let keyCommander = CommandRegistrar()

final class KeyCommandHostingController<Content: View>: UIHostingController<Content> {
    override var canBecomeFirstResponder: Bool { true }
    
    override var keyCommands: [UIKeyCommand]? {
        keyCommander.commands.map { $0.uikit }
    }
    
    override func swiftUIKeyCommand(_ sender: UIKeyCommand?) {
        guard let command = KeyCommand(sender) else { return }
        keyCommander.publisher.send(command)
    }
}
// END:Controller

// START:Extension
extension View {
    func onKeyCommand(
        _ command: KeyCommand,
        perform: @escaping () -> Void
    ) -> some View {
        onReceive(keyCommander.install(command: command).map { _ in () },
                  perform: perform)
    }
    
    func onKeyCommand(
        title: String,
        input: String,
        modifiers: KeyCommand.ModifierFlags = [],
        attributes: KeyCommand.Attributes = [],
        discoverabilityTitle: String? = nil,
        perform: @escaping () -> Void
    ) -> some View {
        let command = KeyCommand(title: title, input: input,
                                 modifierFlags: modifiers,
                                 attributes: attributes,
                                 discoverabilityTitle: discoverabilityTitle)
        return onKeyCommand(command, perform: perform)
    }
}
// END:Extension
